# -----------------------------------------------------------------------
# Copyright: 2010-2022, imec Vision Lab, University of Antwerp
#            2013-2022, CWI, Amsterdam
#
# Contact: astra@astra-toolbox.com
# Website: http://www.astra-toolbox.com/
#
# This file is part of the ASTRA Toolbox.
#
#
# The ASTRA Toolbox is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# The ASTRA Toolbox is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with the ASTRA Toolbox. If not, see <http://www.gnu.org/licenses/>.
#
# -----------------------------------------------------------------------

import astra
import numpy as np
import matplotlib.pyplot as plt
plt.gray()


vol_geom = astra.create_vol_geom(256, 256)
proj_geom = astra.create_proj_geom(
    'parallel', 1.0, 384, np.linspace(0, np.pi, 180, False)
)

# For CPU-based algorithms, a 'projector' object specifies the projection
# model used. In this case, we use the 'strip' model
proj_id = astra.create_projector('strip', proj_geom, vol_geom)

# Create a sinogram from a phantom
phantom_id, P = astra.data2d.shepp_logan(vol_geom)
sinogram_id, sinogram = astra.create_sino(phantom_id, proj_id)

plt.imshow(P)
plt.figure()
plt.imshow(sinogram)

# Create a data object for the reconstruction
rec_id = astra.data2d.create('-vol', vol_geom)

# Set up the parameters for a reconstruction algorithm using the CPU
# The main difference with the configuration of a GPU algorithm is the
# extra ProjectorId setting
cfg = astra.astra_dict('SIRT')
cfg['ReconstructionDataId'] = rec_id
cfg['ProjectionDataId'] = sinogram_id
cfg['ProjectorId'] = proj_id
# Available algorithms:
# ART, SART, SIRT, CGLS, FBP

# Create the algorithm object from the configuration structure
alg_id = astra.algorithm.create(cfg)

# Run 20 iterations of the algorithm
# This will have a runtime in the order of 10 seconds
astra.algorithm.run(alg_id, 20)

# Get the result
rec = astra.data2d.get(rec_id)
plt.figure()
plt.imshow(rec)
plt.show()

# Clean up.
astra.algorithm.delete(alg_id)
astra.data2d.delete(rec_id)
astra.data2d.delete(sinogram_id)
astra.data2d.delete(phantom_id)
astra.projector.delete(proj_id)
